<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\Admin;
use App\Models\Shipment;
use App\Trait\ImageUpload;
use Illuminate\Support\Str;
use App\Enum\ShipmentStatus;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\ShipmentLocation;
use App\Http\Controllers\Controller;
use App\Http\Requests\ShipmentRequest;


class ShipmentController extends Controller
{
    use ImageUpload;
    public function index()
    {
        $admin = Admin::where('id', auth('admin')->user()->id)->first();

        $data = [
            'title' => 'Shipments',
            'shipments' => Shipment::orderBy('id', 'desc')->where('admin_id', auth()->guard('admin')->user()->id)->get(),
            'admin' => $admin
        ];

        return view('dashboard.admin.shipment.index', $data);
    }
    public function create()
    {
        $admin = Admin::where('id', auth('admin')->user()->id)->first();
        $data = [
            'title' => 'Create Shipment',
            'admin' => $admin
        ];

        return view('dashboard.admin.shipment.create', $data);
    }
    public function store(ShipmentRequest $request)
    {
        $data = $request->validated();

        $data['uuid'] = Str::uuid();
        $data['admin_id'] = auth()->guard('admin')->user()->id;
        $data['image'] = $this->uploadImage($request, 'image', 'uploads/shipments/');
        $data['tracking_code'] = getTrackingNumber(config('app.name'));

        Shipment::create($data);

        return redirect()->route('admin.shipment.show', $data['uuid'])->with('success', 'Shipment created successfully!');
    }
    public function show(string $uuid)
    {
        $shipment = Shipment::with('shipmentLocations')->where('uuid', $uuid)->first();
        $admin = Admin::where('id', auth('admin')->user()->id)->first();
        $data = [
            'title' => 'Shipment Details',
            'shipment' => $shipment,
            'shipmentLocations' => $shipment->shipmentLocations()->latest()->get(),
            'shipmentStatues' => ShipmentStatus::cases(),
            'admin' => $admin
        ];

        return view('dashboard.admin.shipment.show', $data);
    }
    public function edit(string $uuid)
    {
        $admin = Admin::where('id', auth('admin')->user()->id)->first();
        $data = [
            'title' => 'Edit Shipment',
            'shipment' => Shipment::with('shipmentLocations')->where('uuid', $uuid)->first(),
            'admin' => $admin
        ];

        return view('dashboard.admin.shipment.edit', $data);
    }
    public function update(ShipmentRequest $request, $uuid)
    {
        $data = $request->validated();

        $shipment = Shipment::where('uuid', $uuid)->first();

        $data['image'] = $this->updateImage($request, 'image', 'uploads/shipments/', $shipment?->image);

        $shipment->update($data);

        return redirect()->route('admin.shipment.show', $shipment->uuid)->with('success', 'Shipment updated successfully!');
    }
    public function delete(string $uuid)
    {
        $shipment = Shipment::where('uuid', $uuid)->first();

        ShipmentLocation::where('shipment_id', $shipment->id)->delete();

        $this->deleteImage('uploads/shipments/', $shipment->image);

        $shipment->delete();

        return redirect()->route('admin.shipment.index')->with('success', 'Shipment deleted successfully!');
    }
    public function print($uuid)
    {
        $shipment = Shipment::with('shipmentLocations')->where('uuid', $uuid)->first();

        getQRCode(route('cargo.tracking.details', $shipment->uuid));

        $data = [
            'shipment' => $shipment,
        ];

        $name = config('app.name') . '-' . 'Receipt';

        $pdf = Pdf::loadView('pdf.shipment', $data);

        if (config('app.env') == 'production') {
            return $pdf->download($name);
        } else {
            return $pdf->stream($name);
        }
    }
}
